<?php

namespace App\Http\Controllers;

use App\Models\DataQueue;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Tymon\JWTAuth\Exceptions\JWTException;
use Tymon\JWTAuth\Exceptions\TokenExpiredException;
use Tymon\JWTAuth\Exceptions\TokenInvalidException;
use Tymon\JWTAuth\Facades\JWTAuth;
use Yajra\DataTables\Facades\DataTables;

class DataQueueController extends Controller
{
    /**
     * @OA\Get(
     *     path="/api/data-queue-list",
     *     tags={"Data Queue"},
     *     summary="Get a list of data queues",
     *     security={{"bearerAuth": {}}},
     *     @OA\Response(
     *         response=200,
     *         description="Successful operation",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", type="array", @OA\Items(ref="#/components/schemas/DataQueue")),
     *         )
     *     )
     * )
     */
    public function fetchDataQueue(Request $request){ //fetchDataQueue

        try {
            $user = JWTAuth::parseToken()->authenticate();
            if (!$user) {
                return ['success' => false, 'message' => 'User not found'];
            }
        } catch (TokenExpiredException $e) {
            return ['success' => false, 'message' => 'Token expired'];
        } catch (TokenInvalidException $e) {
            return ['success' => false, 'message' => 'Token invalid'];
        } catch (JWTException $e) {
            return ['success' => false, 'message' => 'Token absent'];
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'An unexpected error occurred'];
        }
        $user=Auth::user();

        if ($request->has('dT')) {
            $listOfDataQueue = DataQueue::whereNull('deleted_at');
            if($request->fromDate != null && $request->toDate != null){
                $listOfDataQueue = $listOfDataQueue->whereBetween('timestamp', [$request->fromDate.' 00:00:00', $request->toDate.' 23:59:59']);
            }else if($request->fromDate != null){
                $listOfDataQueue = $listOfDataQueue->where('timestamp', '>=', $request->fromDate.' 00:00:00');
            }else if($request->toDate != null){
                $listOfDataQueue = $listOfDataQueue->where('timestamp', '<=', $request->toDate.' 23:59:59');
            }
            if($request->status != null){
                $listOfDataQueue = $listOfDataQueue->where('status', $request->status);
            }
            $listOfDataQueue = $listOfDataQueue->get();
            return DataTables::of($listOfDataQueue)
                        ->setRowId('id')
                        /*->escapeColumns(['action'])*/
                        ->make(true);
        }
        else
        {
            $lists = DataQueue::all();
            return [
                'success' => true,
                'data' => $lists
            ];
        }
    }

    //insert DataQueue with DataQueueAttachment
    /**
     * @OA\Post(
     *     path="/api/data-queue-insert",
     *     tags={"Data Queue"},
     *     summary="Insert a new data queue",
     *     security={{"bearerAuth": {}}},
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             @OA\Property(property="endpoint", type="string", example="endpoint"),
     *             @OA\Property(property="action", type="string", example="action"),
     *             @OA\Property(property="method", type="integer", example="method"),
     *             @OA\Property(property="data", type="string", example="data"),
     *             @OA\Property(property="response", type="string", example="response"),
     *             @OA\Property(property="timestamp", type="string", format="date-time", example="2020-01-01 00:00:00"),
     *             @OA\Property(property="type", type="integer", example=1),
     *             @OA\Property(property="status", type="integer", example=1),
     *             @OA\Property(property="attachment", type="array", @OA\Items(
     *                 @OA\Property(property="attachment", type="string", example="attachment")
     *             )),
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Successful operation",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", ref="#/components/schemas/DataQueue"),
     *         )
     *     )
     * )
     */
    public function insertDataQueue(Request $request){
        $data = [
            'endpoint' => $request->endpoint,
            'action' => $request->action,
            'method' => $request->method,
            'data' => $request->data,
            'response' => $request->response,
            'timestamp' => $request->timestamp?:date('Y-m-d H:i:s'),
            'type' => $request->type?:1,
            'status' => $request->status?:1
        ];
        $dataQueue = DataQueue::create($data);
        $dataQueue->dataQueueAttachment()->createMany($request->attachment);
        return [
            'success' => true,
            'data' => $dataQueue
        ];
    }

    public function dataqueueEdit(Request $request, $id){
        $dataQueue = DataQueue::find($id);
        return [
            'success' => true,
            'data' => $dataQueue
        ];
    }
}
