<?php

namespace App\Http\Controllers;

use App\Models\EnvironmentVariable;
use Illuminate\Http\Request;

class EnvironmentVariableController extends Controller
{
    /**
     * @OA\GET(
     *     path="/api/env-list",
     *     tags={"Environment Variables"},
     *     summary="Get a list of environment variables",
     *     security={{"bearerAuth": {}}},
     *     @OA\Response(
     *         response=200,
     *         description="Successful operation",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", type="array", @OA\Items(ref="#/components/schemas/EnvironmentVariable")),
     *         )
     *     )
     * )
     */
    public function envList(Request $request){
        $listOfEnv = EnvironmentVariable::get();
        return [
            'success' => true,
            'data' => $listOfEnv
        ];
    }
    /**
     * @OA\Post(
     *     path="/api/insert-env",
     *     tags={"Environment Variables"},
     *     summary="Insert a new environment variable",
     *     security={{"bearerAuth": {}}},
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             @OA\Property(property="var_name", type="string", example="VAR_NAME"),
     *             @OA\Property(property="description", type="string", example="Description of the variable"),
     *             @OA\Property(property="status", type="integer", example=1),
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Successful operation",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", ref="#/components/schemas/EnvironmentVariable"),
     *         )
     *     )
     * )
     */
    public function insertenv(Request $request){
        $data = [
            'var_name' => $request->var_name,
            'description' => $request->description,
            'status' => $request->status?:1
        ];
        $env = EnvironmentVariable::create($data);
        return [
            'success' => true,
            'data' => $env
        ];
    }

    /**
     * @OA\GET(
     *     path="/api/env-edit/{id}",
     *     tags={"Environment Variables"},
     *     summary="Get details of a specific environment variable",
     *     security={{"bearerAuth": {}}},
     *     @OA\Parameter(
     *         name="env_id",
     *         in="path",
     *         required=true,
     *         description="ID of the environment variable",
     *         @OA\Schema(type="integer")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Successful operation",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", ref="#/components/schemas/EnvironmentVariable"),
     *         )
     *     )
     * )
     */
    public function envEdit(Request $request,$env_id){
        $data = EnvironmentVariable::find($env_id);
        return [
            'success' => true,
            'data' => $data
        ];
    }

    /**
     * @OA\Post(
     *     path="/api/env-update/{id}",
     *     tags={"Environment Variables"},
     *     summary="Update an existing environment variable",
     *     security={{"bearerAuth": {}}},
     *     @OA\Parameter(
     *         name="env_id",
     *         in="path",
     *         required=true,
     *         description="ID of the environment variable",
     *         @OA\Schema(type="integer")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             @OA\Property(property="var_name", type="string", example="VAR_NAME"),
     *             @OA\Property(property="description", type="string", example="Description of the variable"),
     *             @OA\Property(property="status", type="integer", example=1),
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Successful operation",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", ref="#/components/schemas/EnvironmentVariable"),
     *         )
     *     )
     * )
     */
    public function updateenv(Request $request,$env_id){

        if($env_id != 'bulk'){
            $data = EnvironmentVariable::find($env_id);
            $data->var_name = $request->var_name;
            $data->description = $request->description;
            $data->status = $request->status?:1;
            $data->save();
            return [
                'success' => true,
                'data' => $data
            ];
        }else{
            if($request->envs && count($request->envs) > 0){
                foreach($request->envs as $env){
                    $data = EnvironmentVariable::find($env['id']);
                    $data->var_name = $env['var_name'];
                    $data->description = $env['description'];
                    $data->status = $env['status'];
                    $data->save();
                }
            }
            return [
                'success' => true
            ];
        }
    }


    /**
     * @OA\Delete(
     *     path="/api/env-delete/{id}",
     *     tags={"Environment Variables"},
     *     summary="Delete an existing environment variable",
     *     security={{"bearerAuth": {}}},
     *     @OA\Parameter(
     *         name="env_id",
     *         in="path",
     *         required=true,
     *         description="ID of the environment variable",
     *         @OA\Schema(type="integer")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Successful operation",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", ref="#/components/schemas/EnvironmentVariable"),
     *         )
     *     )
     * )
     */
    public function delenv(Request $request,$env_id){
        $data = EnvironmentVariable::find($env_id);
        $data->delete();
        return [
            'success' => true,
            'data' => $data
        ];
    }

    public function envApiInsertExisting(){
        $data = $data = [
            "date_time"=>true, // can select past future dates
            "date_range"=>true,
            "auto_en_route"=>false,
            "navigation"=>true,
            "show_only_today_stops"=>true,
            "stop_display_restriction"=>0,
            "first_stop_visible"=>true,
            "auto_stop_proximity"=>true,
            "can_reject_stop"=>true,
            "can_transfer_stop"=>true,
            "group_stop"=>false,
            "call_client"=>true,
            "chat_client"=>true,
            "patient_signature"=>true,
            "print_requisition"=>true,
            "missed_draw"=>true,
            "add_edit_insurance"=>true,
            "add_edit_test"=>true,
            "add_edit_specimen"=>true,
            "specimen_barcode"=>true,
            "edit_provider"=>true,
            "edit_dx"=>true,
            "request_return_visit"=>true,
            "preview_image"=>true,
            "add_notes"=>true,
            "add_file"=>true,
            "facility_collection"=>true,
            "facility_add_edit_insurance"=>true,
            "facility_add_edit_specimen"=>true,
            "facility_specimen_barcode"=>true,
            "facility_edit_provider"=>true,
            "facility_edit_dx"=>true,
            "facility_request_return_visit"=>true,
            "facility_preview_image"=>true,
            "facility_add_notes"=>true,
            "facility_add_file"=>true,
            "deliver_to_lab"=>true,
            "transfer_technician"=>true,
            "skip_drop_off"=>true,
            "technician_messenger"=>true,
            "show_collection_by_date_range"=>true,
            "show_fasting_icon"=>true,
            "schedule_timer"=>false,
            'end_shift_notes_required'=>true,
            'hide_deliver_to_lab'=>true,
            'camera_icon_rejection'=>true,
            'camera_icon_order_details'=>true,
            'gallery_access_in_app'=>true,
            'end_shift_offline'=>true,
            'imaging_module'=>true

        ];
        $insertCount = 0;
        $updateCount = 0;
        foreach ($data as $key => $value) {
            $env = EnvironmentVariable::where('var_name',$key)->first();
            if(!$env){
                $env = new EnvironmentVariable();
                $insertCount++;
            }else{
                $updateCount++;
            }
            $env->var_name = $key;
            $env->description = $value;
            $env->status = $value==true?1:0;
            $env->save();
        }

        return [
            'success' => true,
            'insertCount' => $insertCount,
            'updateCount' => $updateCount
        ];
    }

    /**
     * @OA\Get(
     *     path="/get-env",
     *     summary="Get environment settings for the technician",
     *     tags={"Environment Variables"},
     *     description="This endpoint returns various environment settings for the technician.",
     *     operationId="techEnvUpdated",
     *     security={{"bearerAuth": {}}},
     *     @OA\Response(
     *         response=200,
     *         description="Successful operation",
     *
     *             @OA\JsonContent(
     *                type="object",
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="date_time", type="boolean", example=true),
     *                 @OA\Property(property="date_range", type="boolean", example=true),
     *                 @OA\Property(property="auto_en_route", type="boolean", example=false),
     *                 @OA\Property(property="navigation", type="boolean", example=true),
     *                 @OA\Property(property="show_only_today_stops", type="boolean", example=true),
     *                 @OA\Property(property="stop_display_restriction", type="integer", example=0),
     *                 @OA\Property(property="first_stop_visible", type="boolean", example=true),
     *                 @OA\Property(property="auto_stop_proximity", type="boolean", example=true),
     *                 @OA\Property(property="can_reject_stop", type="boolean", example=true),
     *                 @OA\Property(property="can_transfer_stop", type="boolean", example=true),
     *                 @OA\Property(property="group_stop", type="boolean", example=false),
     *                 @OA\Property(property="call_client", type="boolean", example=true),
     *                 @OA\Property(property="chat_client", type="boolean", example=true),
     *                 @OA\Property(property="patient_signature", type="boolean", example=true),
     *                 @OA\Property(property="print_requisition", type="boolean", example=true),
     *                 @OA\Property(property="missed_draw", type="boolean", example=true),
     *                 @OA\Property(property="add_edit_insurance", type="boolean", example=true),
     *                 @OA\Property(property="add_edit_test", type="boolean", example=true),
     *                 @OA\Property(property="add_edit_specimen", type="boolean", example=true),
     *                 @OA\Property(property="specimen_barcode", type="boolean", example=true),
     *                 @OA\Property(property="edit_provider", type="boolean", example=true),
     *                 @OA\Property(property="edit_dx", type="boolean", example=true),
     *                 @OA\Property(property="request_return_visit", type="boolean", example=true),
     *                 @OA\Property(property="preview_image", type="boolean", example=true),
     *                 @OA\Property(property="add_notes", type="boolean", example=true),
     *                 @OA\Property(property="add_file", type="boolean", example=true),
     *                 @OA\Property(property="facility_collection", type="boolean", example=true),
     *                 @OA\Property(property="facility_add_edit_insurance", type="boolean", example=true),
     *                 @OA\Property(property="facility_add_edit_specimen", type="boolean", example=true),
     *                 @OA\Property(property="facility_specimen_barcode", type="boolean", example=true),
     *                 @OA\Property(property="facility_edit_provider", type="boolean", example=true),
     *                 @OA\Property(property="facility_edit_dx", type="boolean", example=true),
     *                 @OA\Property(property="facility_request_return_visit", type="boolean", example=true),
     *                 @OA\Property(property="facility_preview_image", type="boolean", example=true),
     *                 @OA\Property(property="facility_add_notes", type="boolean", example=true),
     *                 @OA\Property(property="facility_add_file", type="boolean", example=true),
     *                 @OA\Property(property="deliver_to_lab", type="boolean", example=true),
     *                 @OA\Property(property="transfer_technician", type="boolean", example=true),
     *                 @OA\Property(property="skip_drop_off", type="boolean", example=true),
     *                 @OA\Property(property="technician_messenger", type="boolean", example=true),
     *                 @OA\Property(property="show_collection_by_date_range", type="boolean", example=true),
     *                 @OA\Property(property="show_fasting_icon", type="boolean", example=true),
     *                 @OA\Property(property="schedule_timer", type="boolean", example=false),
     *                 @OA\Property(property="imaging_module", type="boolean", example=false)
     *
     *             )
     *         )
     *     )
     *   )
     *
     */
    public function fetchEnv(){
        $data = EnvironmentVariable::get([
            'var_name',
            'description',
            'status'
        ]);
        $result = [];
        foreach ($data as $item) {
            if($item['var_name']!='stop_display_restriction') {
                if($item['var_name']=='dashboard_date_range')
                    $result[$item['var_name']] = $item['status'] == 1 ? $item['description'] : false;
                else if($item['var_name']=='order_display_mode')
                    $result[$item['var_name']] = $item['status'] == 1 ? (int) $item['description'] : 1; //default: flat order
                else
                    $result[$item['var_name']] = $item['status'] == 1;
            } else {
                $result[$item['var_name']] = $item['status'];
            }
        }
        $result['timezone'] ="America/Chicago";
        // $result['v5_baseurl'] ="https://poc.edevlabwerks.com/";
        $result['v5_baseurl'] = env('MAIN_APP_RUNNING',true) ? 'https://sys.edevlabwerks.com/' : 'https://poc.edevlabwerks.com/';

        return [
            'success' => true,
            'data' => $result
        ];
    }
    function test()
    {
        return response()->json(["success"=>true,"data"=>["dummy"=>true]],200);
    }
}
