<?php

namespace App\Http\Controllers;

use App\Models\Appointmenttime;
use Carbon\Carbon;
use Illuminate\Http\Request;

class AppointmenttimeController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:api');
    }

    /**
     * @OA\Get(
     *     path="/api/appointmenttime/getlist",
     *     summary="Get a list of appointment times",
     *     description="Fetches a list of available appointment times with their details",
     *     operationId="getList",
     *     tags={"Appointment Times"},
     *     @OA\Response(
     *         response=200,
     *         description="Successful operation",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(
     *                 property="success",
     *                 type="boolean",
     *                 example=true
     *             ),
     *             @OA\Property(
     *                 property="added",
     *                 type="array",
     *                 @OA\Items(
     *                     type="object",
     *                     @OA\Property(
     *                         property="appointmenttime_id",
     *                         type="integer",
     *                         example=5
     *                     ),
     *                     @OA\Property(
     *                         property="appointment_time",
     *                         type="string",
     *                         example="9:00AM - 12:00PM"
     *                     ),
     *                     @OA\Property(
     *                         property="created_at",
     *                         type="string",
     *                         format="date-time",
     *                         example="2021-03-10T14:58:56.000000Z"
     *                     ),
     *                     @OA\Property(
     *                         property="updated_at",
     *                         type="string",
     *                         format="date-time",
     *                         nullable=true,
     *                         example=null
     *                     ),
     *                     @OA\Property(
     *                         property="deleted_at",
     *                         type="string",
     *                         format="date-time",
     *                         nullable=true,
     *                         example=null
     *                     )
     *                 )
     *             ),
     *             @OA\Property(
     *                 property="updated",
     *                 type="array",
     *                 @OA\Items(
     *                     type="object"
     *                 )
     *             ),
     *             @OA\Property(
     *                 property="deleted",
     *                 type="array",
     *                 @OA\Items(
     *                     type="object"
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=400,
     *         description="Bad Request"
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Server Error"
     *     )
     * )
     */

    public function getList(Request $request)
    {
        $currentTimeStamp = $request->timestamp ? Carbon::createFromTimestamp($request->timestamp)->format('Y-m-d') : null;
        $added = [];
        $updated = [];
        $deleted = [];
        if ($currentTimeStamp) {
            $added = Appointmenttime::whereNull('appointmenttime.deleted_at')
                ->whereDate('appointmenttime.created_at', '>=', $currentTimeStamp)
                ->get([
                    'appointmenttime.appointmenttime_id',
                    'appointmenttime.appointment_time',
                    'appointmenttime.created_at',
                    'appointmenttime.updated_at',
                    'appointmenttime.deleted_at'
                ]);

            $addedIDs = $added->pluck('appointmenttime.appointmenttime_id')->toArray();
            $updated = Appointmenttime::whereNotIn('appointmenttime.appointmenttime_id', $addedIDs)
                ->whereNull('appointmenttime.deleted_at')
                ->whereDate('appointmenttime.updated_at', '>=', $currentTimeStamp)
                ->get([
                    'appointmenttime.appointmenttime_id',
                    'appointmenttime.appointment_time',
                    'appointmenttime.created_at',
                    'appointmenttime.updated_at',
                    'appointmenttime.deleted_at'
                ]);

            $deleted = Appointmenttime::whereNotNull('appointmenttime.deleted_at')
                ->whereDate('appointmenttime.updated_at', '>=', $currentTimeStamp)
                ->get([
                    'appointmenttime.appointmenttime_id',
                    'appointmenttime.appointment_time',
                    'appointmenttime.created_at',
                    'appointmenttime.updated_at',
                    'appointmenttime.deleted_at'
                ]);
        } else {
            $added = appointmenttime::whereNull('appointmenttime.deleted_at')
                ->get([
                    'appointmenttime.appointmenttime_id',
                    'appointmenttime.appointment_time',
                    'appointmenttime.created_at',
                    'appointmenttime.updated_at',
                    'appointmenttime.deleted_at'
                ]);
            $deleted = appointmenttime::whereNotNull('appointmenttime.deleted_at')
                ->get([
                    'appointmenttime.appointmenttime_id',
                    'appointmenttime.appointment_time',
                    'appointmenttime.created_at',
                    'appointmenttime.updated_at',
                    'appointmenttime.deleted_at'
                ]);
        }
        return response()->json(['success' => true, "added" => $added, "updated" => $updated, "deleted" => $deleted], 200);
    }
}
